---
title: Sandbox options · Cloudflare Sandbox SDK docs
description: Configure sandbox behavior by passing options when creating a
  sandbox instance with getSandbox().
lastUpdated: 2025-11-06T16:25:37.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/sandbox/configuration/sandbox-options/
  md: https://developers.cloudflare.com/sandbox/configuration/sandbox-options/index.md
---

Configure sandbox behavior by passing options when creating a sandbox instance with `getSandbox()`.

## Available options

```ts
import { getSandbox } from '@cloudflare/sandbox';


const sandbox = getSandbox(binding, sandboxId, options?: SandboxOptions);
```

### keepAlive

**Type**: `boolean` **Default**: `false`

Keep the container alive indefinitely by preventing automatic shutdown. When `true`, the container will never auto-timeout and must be explicitly destroyed using `destroy()`.

* JavaScript

  ```js
  // For long-running processes that need the container to stay alive
  const sandbox = getSandbox(env.Sandbox, "user-123", {
    keepAlive: true,
  });


  // Run your long-running process
  await sandbox.startProcess("python long_running_script.py");


  // Important: Must explicitly destroy when done
  try {
    // Your work here
  } finally {
    await sandbox.destroy(); // Required to prevent containers running indefinitely
  }
  ```

* TypeScript

  ```ts
  // For long-running processes that need the container to stay alive
  const sandbox = getSandbox(env.Sandbox, 'user-123', {
    keepAlive: true
  });


  // Run your long-running process
  await sandbox.startProcess('python long_running_script.py');


  // Important: Must explicitly destroy when done
  try {
    // Your work here
  } finally {
    await sandbox.destroy(); // Required to prevent containers running indefinitely
  }
  ```

Resource management with keepAlive

When `keepAlive: true` is set, containers will not automatically timeout. They must be explicitly destroyed using `destroy()` to prevent containers running indefinitely and counting toward your account limits.

## When to use keepAlive

Use `keepAlive: true` for:

* **Long-running builds** - CI/CD pipelines that may have idle periods between steps
* **Batch processing** - Jobs that process data in waves with gaps between batches
* **Monitoring tasks** - Processes that periodically check external services
* **Interactive sessions** - User-driven workflows where the container should remain available

By default, containers automatically shut down after 10 minutes of inactivity. The `keepAlive` option prevents this automatic shutdown.

## Related resources

* [Background processes guide](https://developers.cloudflare.com/sandbox/guides/background-processes/) - Using `keepAlive` with long-running processes
* [Lifecycle API](https://developers.cloudflare.com/sandbox/api/lifecycle/) - Create and manage sandboxes
* [Sandboxes concept](https://developers.cloudflare.com/sandbox/concepts/sandboxes/) - Understanding sandbox lifecycle
